<?php
/**
 * Polyfills for PHP 5.4-8.4 compatibility
 * Provides backward-compatible functions for PHP 5.4-5.5
 * 
 * @package Social Boost
 * @since 3.5.3
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * hash_equals() polyfill for PHP < 5.6
 * Timing attack safe string comparison
 * 
 * @link https://www.php.net/manual/en/function.hash-equals.php
 * @param string $known_string The string of known length to compare against
 * @param string $user_string The user-supplied string
 * @return bool Returns true when the two strings are equal, false otherwise
 */
if (!function_exists('hash_equals')) {
    function hash_equals($known_string, $user_string) {
        if (!is_string($known_string)) {
            trigger_error('hash_equals(): Expected known_string to be a string', E_USER_WARNING);
            return false;
        }
        
        if (!is_string($user_string)) {
            trigger_error('hash_equals(): Expected user_string to be a string', E_USER_WARNING);
            return false;
        }
        
        $known_len = strlen($known_string);
        $user_len = strlen($user_string);
        
        if ($known_len !== $user_len) {
            return false;
        }
        
        $result = 0;
        for ($i = 0; $i < $known_len; $i++) {
            $result |= ord($known_string[$i]) ^ ord($user_string[$i]);
        }
        
        return $result === 0;
    }
}

/**
 * Safe session start wrapper for PHP 5.4-8.4
 * Properly checks session status and headers before starting session
 * 
 * @return bool True if session started or already active, false otherwise
 */
if (!function_exists('amsb_session_start')) {
    function amsb_session_start() {
        // Check if headers already sent
        if (headers_sent()) {
            return false;
        }
        
        // For PHP 5.4+ use session_status()
        if (function_exists('session_status')) {
            if (session_status() === PHP_SESSION_ACTIVE) {
                return true;
            }
            if (session_status() === PHP_SESSION_DISABLED) {
                return false;
            }
            return @session_start();
        }
        
        // Fallback for older PHP (pre-5.4, but we support 5.4+)
        if (session_id()) {
            return true;
        }
        
        return @session_start();
    }
}

/**
 * Safe JSON decode with error handling
 * Returns default value if JSON is invalid
 * 
 * @param string $json The JSON string to decode
 * @param bool $assoc Return associative array when true
 * @param mixed $default Default value to return on failure
 * @return mixed Decoded value or default on failure
 */
if (!function_exists('amsb_json_decode')) {
    function amsb_json_decode($json, $assoc = true, $default = array()) {
        if (empty($json) || !is_string($json)) {
            return $default;
        }
        
        $result = json_decode($json, $assoc);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return $default;
        }
        
        return $result !== null ? $result : $default;
    }
}

/**
 * Safe strtolower wrapper for PHP 8.0+ compatibility
 * Handles null values without throwing errors
 * 
 * @param mixed $string The string to convert to lowercase
 * @return string Lowercase string or empty string if input is invalid
 */
if (!function_exists('amsb_strtolower')) {
    function amsb_strtolower($string) {
        if ($string === null || $string === false) {
            return '';
        }
        
        return strtolower((string)$string);
    }
}

/**
 * Check if WC_VERSION constant is defined and optionally compare versions
 * 
 * @param string $version Version to compare against (optional)
 * @param string $operator Comparison operator (optional)
 * @return bool|mixed
 */
if (!function_exists('amsb_check_wc_version')) {
    function amsb_check_wc_version($version = null, $operator = '>=') {
        if (!defined('WC_VERSION')) {
            return false;
        }
        
        if ($version === null) {
            return true;
        }
        
        return version_compare(WC_VERSION, $version, $operator);
    }
}

// =============================================================================
// FUTURE-PROOF STRING FUNCTION WRAPPERS (PHP 8.1+ null deprecation)
// =============================================================================

/**
 * Safe strlen() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to strlen() is deprecated
 * 
 * @param mixed $string The string to measure
 * @return int Length of the string, or 0 if invalid
 */
if (!function_exists('amsb_strlen')) {
    function amsb_strlen($string) {
        if ($string === null || $string === false) {
            return 0;
        }
        return strlen((string)$string);
    }
}

/**
 * Safe trim() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to trim() is deprecated
 * 
 * @param mixed $string The string to trim
 * @param string $characters Characters to trim (optional)
 * @return string Trimmed string or empty string if invalid
 */
if (!function_exists('amsb_trim')) {
    function amsb_trim($string, $characters = " \t\n\r\0\x0B") {
        if ($string === null || $string === false) {
            return '';
        }
        return trim((string)$string, $characters);
    }
}

/**
 * Safe substr() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to substr() is deprecated
 * 
 * @param mixed $string The input string
 * @param int $offset Start position
 * @param int|null $length Length (optional)
 * @return string|false Substring or false on failure
 */
if (!function_exists('amsb_substr')) {
    function amsb_substr($string, $offset, $length = null) {
        if ($string === null || $string === false) {
            return '';
        }
        if ($length === null) {
            return substr((string)$string, $offset);
        }
        return substr((string)$string, $offset, $length);
    }
}

/**
 * Safe str_replace() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to str_replace() is deprecated
 * 
 * @param mixed $search Value to search for
 * @param mixed $replace Value to replace with
 * @param mixed $subject String or array to search in
 * @param int|null &$count Number of replacements performed (optional)
 * @return string|array Replaced string/array
 */
if (!function_exists('amsb_str_replace')) {
    function amsb_str_replace($search, $replace, $subject, &$count = null) {
        // Handle null values
        if ($subject === null || $subject === false) {
            return '';
        }
        if ($search === null) {
            $search = '';
        }
        if ($replace === null) {
            $replace = '';
        }
        if ($count !== null) {
            return str_replace($search, $replace, $subject, $count);
        }
        return str_replace($search, $replace, $subject);
    }
}

/**
 * Safe strpos() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to strpos() is deprecated
 * 
 * @param mixed $haystack The string to search in
 * @param mixed $needle The string to search for
 * @param int $offset Start position (optional)
 * @return int|false Position of needle or false if not found
 */
if (!function_exists('amsb_strpos')) {
    function amsb_strpos($haystack, $needle, $offset = 0) {
        if ($haystack === null || $haystack === false || $needle === null || $needle === false) {
            return false;
        }
        return strpos((string)$haystack, (string)$needle, $offset);
    }
}

/**
 * Safe htmlspecialchars() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to htmlspecialchars() is deprecated
 * 
 * @param mixed $string The string to convert
 * @param int $flags Flags (optional)
 * @param string|null $encoding Character encoding (optional)
 * @param bool $double_encode Double encode (optional)
 * @return string Converted string
 */
if (!function_exists('amsb_htmlspecialchars')) {
    function amsb_htmlspecialchars($string, $flags = ENT_QUOTES | ENT_SUBSTITUTE, $encoding = null, $double_encode = true) {
        if ($string === null || $string === false) {
            return '';
        }
        if ($encoding === null) {
            $encoding = 'UTF-8';
        }
        return htmlspecialchars((string)$string, $flags, $encoding, $double_encode);
    }
}

/**
 * Safe stripslashes() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to stripslashes() is deprecated
 * 
 * @param mixed $string The string to process
 * @return string String with backslashes stripped
 */
if (!function_exists('amsb_stripslashes')) {
    function amsb_stripslashes($string) {
        if ($string === null || $string === false) {
            return '';
        }
        return stripslashes((string)$string);
    }
}

/**
 * Safe explode() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to explode() is deprecated
 * 
 * @param string $separator The boundary string
 * @param mixed $string The input string
 * @param int $limit Limit (optional)
 * @return array Array of strings
 */
if (!function_exists('amsb_explode')) {
    function amsb_explode($separator, $string, $limit = PHP_INT_MAX) {
        if ($string === null || $string === false) {
            return array();
        }
        if ($separator === null || $separator === false || $separator === '') {
            return array((string)$string);
        }
        return explode($separator, (string)$string, $limit);
    }
}

/**
 * Safe implode() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to implode() is deprecated
 * Also validates array parameter
 * Supports both implode($glue, $array) and implode($array) syntax
 * 
 * @param string|array $separator The boundary string or array (if single argument)
 * @param mixed $array The array to join (optional)
 * @return string Joined string
 */
if (!function_exists('amsb_implode')) {
    function amsb_implode($separator, $array = null) {
        // Handle single argument: implode($array)
        if ($array === null) {
            $array = $separator;
            $separator = '';
        }
        
        if (!is_array($array)) {
            return '';
        }
        if ($separator === null) {
            $separator = '';
        }
        return implode((string)$separator, $array);
    }
}

/**
 * Safe strtotime() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to strtotime() is deprecated
 * 
 * @param mixed $datetime The datetime string
 * @param int|null $baseTimestamp Base timestamp (optional)
 * @return int|false Timestamp or false on failure
 */
if (!function_exists('amsb_strtotime')) {
    function amsb_strtotime($datetime, $baseTimestamp = null) {
        if ($datetime === null || $datetime === false || $datetime === '') {
            return false;
        }
        if ($baseTimestamp === null) {
            return strtotime((string)$datetime);
        }
        return strtotime((string)$datetime, $baseTimestamp);
    }
}

/**
 * Safe preg_match() wrapper for PHP 8.1+ compatibility
 * PHP 8.1+: Passing null to preg_match() is deprecated
 * 
 * @param string $pattern The pattern to search for
 * @param mixed $subject The input string
 * @param array|null &$matches Array for matches (optional)
 * @param int $flags Flags (optional)
 * @param int $offset Offset (optional)
 * @return int|false Number of matches or false on failure
 */
if (!function_exists('amsb_preg_match')) {
    function amsb_preg_match($pattern, $subject, &$matches = null, $flags = 0, $offset = 0) {
        if ($subject === null || $subject === false) {
            return 0;
        }
        if ($pattern === null || $pattern === false) {
            return false;
        }
        return preg_match($pattern, (string)$subject, $matches, $flags, $offset);
    }
}

// =============================================================================
// PHP 7.3+ POLYFILLS (Backward Compatibility)
// =============================================================================

/**
 * Polyfill for array_key_first() (PHP 7.3+)
 * 
 * @param array $array Input array
 * @return mixed|null First key or null if array is empty
 */
if (!function_exists('array_key_first')) {
    function array_key_first(array $array) {
        foreach ($array as $key => $value) {
            return $key;
        }
        return null;
    }
}

/**
 * Polyfill for array_key_last() (PHP 7.3+)
 * 
 * @param array $array Input array
 * @return mixed|null Last key or null if array is empty
 */
if (!function_exists('array_key_last')) {
    function array_key_last(array $array) {
        if (empty($array)) {
            return null;
        }
        return key(array_slice($array, -1, 1, true));
    }
}

/**
 * Polyfill for str_contains() (PHP 8.0+)
 * Safe version that handles null parameters
 * 
 * @param mixed $haystack The string to search in
 * @param mixed $needle The string to search for
 * @return bool True if needle is found in haystack
 */
if (!function_exists('str_contains')) {
    function str_contains($haystack, $needle) {
        if ($haystack === null || $needle === null) {
            return false;
        }
        return strpos((string)$haystack, (string)$needle) !== false;
    }
}

/**
 * Polyfill for str_starts_with() (PHP 8.0+)
 * 
 * @param mixed $haystack The string to check
 * @param mixed $needle The substring to check for
 * @return bool True if haystack starts with needle
 */
if (!function_exists('str_starts_with')) {
    function str_starts_with($haystack, $needle) {
        if ($haystack === null || $needle === null) {
            return false;
        }
        $haystack = (string)$haystack;
        $needle = (string)$needle;
        return strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

/**
 * Polyfill for str_ends_with() (PHP 8.0+)
 * 
 * @param mixed $haystack The string to check
 * @param mixed $needle The substring to check for
 * @return bool True if haystack ends with needle
 */
if (!function_exists('str_ends_with')) {
    function str_ends_with($haystack, $needle) {
        if ($haystack === null || $needle === null) {
            return false;
        }
        $haystack = (string)$haystack;
        $needle = (string)$needle;
        $length = strlen($needle);
        if ($length === 0) {
            return true;
        }
        return substr($haystack, -$length) === $needle;
    }
}

/**
 * Safe wrapper for in_array() - PHP 8.0+ requires array as haystack
 * Prevents TypeError when haystack is not an array
 */
if (!function_exists('amsb_in_array')) {
    function amsb_in_array($needle, $haystack, $strict = false) {
        if (!is_array($haystack)) {
            return false;
        }
        return in_array($needle, $haystack, $strict);
    }
}

