import React, { Component } from 'react';
import ReactTable from 'react-table';
import 'react-table/react-table.css';

import '../../../css/admin/shared/table.scss';
import '../../../css/admin/manage/links.scss';

import Api from '../../shared/Api';
import Columns from './Columns';
import SelectColumns from '../general/SelectColumns';
import Totals from '../general/Totals';

export default class Links extends Component {
    constructor(props) {
        super(props);

        // Use default selectedColumns or restore from LocalStorage.
        let selectedColumns = ['categories', 'name', 'clicks', 'shortlink', 'url'];
        let savedSelectedColumns = localStorage.getItem( 'eafl-admin-manage-links-columns');

        if (savedSelectedColumns) {
            savedSelectedColumns = JSON.parse(savedSelectedColumns);

            if (Array.isArray(savedSelectedColumns)) {
                selectedColumns = savedSelectedColumns;
            }
        }

        this.state = {
            data: [],
            pages: null,
            filtered: false,
            total: false,
            loading: true,
            columns: Columns.getColumns( this ),
            selectedColumns: selectedColumns,
            selectedRows: {},
            selectedAllRows: 0,
        }

        this.refreshData = this.refreshData.bind(this);
        this.fetchData = this.fetchData.bind(this);
        this.toggleSelectRow = this.toggleSelectRow.bind(this);
        this.toggleSelectAll = this.toggleSelectAll.bind(this);
        this.getSelectedRows = this.getSelectedRows.bind(this);
        this.onColumnsChange = this.onColumnsChange.bind(this);
    }

    toggleSelectRow(id) {
        let newSelected = { ...this.state.selectedRows };

        newSelected[id] = !newSelected[id];

        const nbrSelected = Object.values(newSelected).filter(value => value).length;
        let selectedAllRows = 2;

        if ( 0 === nbrSelected ) {
            selectedAllRows = 0;
        } else if ( this.state.data.length === nbrSelected ) {
            selectedAllRows = 1;
        }

        this.setState({
            selectedRows: newSelected,
            selectedAllRows,
        });
    }

    toggleSelectAll() {
        let newSelected = {};

        if ( 0 === this.state.selectedAllRows ) {
            for ( let row of this.state.data ) {
                newSelected[row.id] = true;
            }
        }

        this.setState({
            selectedRows: newSelected,
            selectedAllRows: 0 === this.state.selectedAllRows ? 1 : 0,
        });
    }

    getSelectedRows() {
        return Object.keys(this.state.selectedRows).filter(id => this.state.selectedRows[id]).map(id => parseInt(id));
    }

    refreshData() {
        this.refReactTable.fireFetchData();
    }

    fetchData(state, instance) {
        this.setState({ loading: true });
        const currentData = state.data;

        Api.getLinks({
            pageSize: state.pageSize,
            page: state.page,
            sorted: state.sorted,
            filtered: state.filtered,
        }).then(data => {
            if ( data ) {
                let newState = {
                    data: data.links,
                    pages: data.pages,
                    filtered: data.filtered,
                    total: data.total,
                    loading: false,
                };

                if ( JSON.stringify( data.links.map( row => row.id ) ) !== JSON.stringify( currentData.map( row => row.id )  ) ) {
                    newState.selectedRows = {};
                    newState.selectedAllRows = 0;
                }

                this.setState(newState);
            }
        });
    }

    onColumnsChange(id, checked) {
        let selectedColumns = [ ...this.state.selectedColumns ];

        if (checked) {
            selectedColumns.push(id);
        } else {
            selectedColumns = selectedColumns.filter(c => c !== id);
        }

        this.setState({
            selectedColumns
        });

        localStorage.setItem( 'eafl-admin-manage-links-columns', JSON.stringify( selectedColumns ) );
    }

    render() {
        const { data, pages, loading } = this.state;
        const selectedColumns = this.state.columns.filter(column => 'actions' === column.id || this.state.selectedColumns.includes(column.id));

        return (
            <div className="eafl-admin-manage-page">
                <div className="eafl-admin-manage-header">
                    <SelectColumns
                        onColumnsChange={this.onColumnsChange}
                        columns={this.state.columns}
                        selectedColumns={this.state.selectedColumns}
                    />
                    {
                        this.state.selectedColumns.includes( 'bulk_edit' )
                        &&
                        <button
                            className="button"
                            onClick={ () => {
                                EAFL_Modal.open('bulk-edit', { linkIds: this.getSelectedRows(), saveCallback: () => this.refreshData() });
                            }}
                            disabled={ 0 === this.getSelectedRows().length }
                        >{ eafl_admin_manage_modal.text.manage.links_bulk_edit } { this.getSelectedRows().length } { eafl_admin_manage_modal.text.manage.links }...</button>
                    }
                    <button
                        className="button button-primary"
                        onClick={() => {
                            EAFL_Modal.open('create', { saveCallback: () => this.refreshData() });
                        }}
                    >{ eafl_admin_manage_modal.text.manage.links_create }</button>
                </div>
                <div className="eafl-admin-manage-table-container">
                    <Totals
                        filtered={this.state.filtered}
                        total={this.state.total}
                    />
                    <ReactTable
                        ref={(refReactTable) => {this.refReactTable = refReactTable;}}
                        manual
                        columns={selectedColumns}
                        data={data}
                        pages={pages}
                        loading={loading}
                        onFetchData={this.fetchData}
                        defaultPageSize={25}
                        defaultSorted={[{
                            id: "id",
                            desc: true
                        }]}
                        filterable
                        resizable={false}
                        className="eafl-admin-manage-table eafl-admin-table -highlight"
                    />
                </div>
            </div>
        );
    }
}